;*****************************************************************************        
;
;   Module:     main.asm
;               
;   Author:     Mike Hibbett 
;                                                                  
;   Version:    1.0 5/10/03                                                  
;
;               Main source file for EPE Car Computer project
;               This file includes all other source files.
;
;*****************************************************************************        

    list p=16f877, st=OFF, x=OFF, n=0
    errorlevel -302
    errorlevel -306

    #include <p16f877.inc>

    __config  0x3ffa

    ; Pull in variable definitions and general constants
    #include "equates.inc"


;*****************************************************************************        
;
;   Function :  Reset vector
;               Hardware entry point to the code
;               This vector is also used for soft resets in the event of 
;               abnormal events
;
;   Input:      None.
;
;   Output:     N/A
;
;*****************************************************************************        
    
    ORG    0

RESET        GOTO    Main


;*****************************************************************************        
;
;   Function :  Interrupt vector
;               Hardware entry point for interrupts
;               This vector handles all external and internal interrupts
;
;   Input:      None.
;
;   Output:     N/A
;
;*****************************************************************************        
    
    ORG    4
    
INTERRUPT
    movwf   W_TEMP
    swapf   STATUS, W
    clrf    STATUS
    movwf   STATUS_TEMP    
    movfw   PCLATH
    movwf   PCLATH_TEMP
    clrf    PCLATH
    
    btfss   INTCON, RBIF
    goto    trytimerInt
    
    bcf     INTCON, RBIF
    bsf     flags1, INT_KEYPRESS
    
trytimerInt
    btfss   PIR1, TMR1IF
    goto    intDone
    
    clrf    TMR1L
    movlw   0x010
    movwf   TMR1H                   ; Restart the timer

    bcf     PIR1, TMR1IF            ; Must clear interrupt flag

    incf    kbdTimeout, F              

    incf    intTick,F
    btfss   intTick, 2              ; Have we counted to 4?
    goto    intDone                 ; ;No
    
    bsf     flags1, INT_RTC_TICK    ; Tell foreground task that clock update occurred
    clrf    intTick
    
    incf    minutesBCD, F
    movfw   minutesBCD
    andlw   0x0F
    sublw   0x0A
    btfss   STATUS,Z
    goto    intDone
    movfw   minutesBCD

    andlw   0x0F0
    addlw   0x010
    movwf   minutesBCD
    sublw   0x060
    btfss   STATUS,Z
    goto    intDone
    clrf    minutesBCD
    
    incf    hoursBCD, F
    movfw   hoursBCD
    andlw   0x0F
    sublw   0x0A
    btfss   STATUS,Z
    goto    tst24
    movfw   hoursBCD

    andlw   0x0F0
    addlw   0x010
    movwf   hoursBCD
    
tst24
    movfw   hoursBCD
    sublw   0x024
    btfss   STATUS,Z
    goto    intDone
    clrf    hoursBCD

    incf    daysBCD, F
    movfw   daysBCD
    andlw   0x0F
    sublw   0x0A
    btfss   STATUS,Z
    goto    tstFeb
    movfw   daysBCD

    andlw   0x0F0
    addlw   0x010
    movwf   daysBCD
    
tstFeb
    ; Check for day overflow. This needs month & leap year calculations
    call    getDaysInMonthBCD       ; Return value in W 
    movwf   daysInMonthBCD
    call    adjustForLeapYear       ; Return value in W 
    addwf   daysInMonthBCD, F       ; feb leap year = 0x29
    
    movfw   daysBCD
    sublw   0x30
    btfsc   STATUS, Z
    goto    tf001   
    decf    daysBCD, W
    goto    tf002
    
tf001
    movlw   0x29

tf002
    subwf   daysInMonthBCD, W
    btfss   STATUS, Z
    goto    intDone
    
    clrf    daysBCD
    incf    daysBCD, F
    
    ; Every month, apply the minutes offset ( which is always added )
    movfw   minOffset
    movwf   minutesBCD
    
    incf    monthsBCD, F
    movfw   monthsBCD
    andlw   0x0F
    sublw   0x0A
    btfss   STATUS,Z
    goto    tstMonth
    movfw   monthsBCD

    andlw   0x0F0
    addlw   0x010
    movwf   monthsBCD

tstMonth
    movfw   monthsBCD
    sublw   0x013
    btfss   STATUS,Z
    goto    intDone
    clrf    monthsBCD
    incf    monthsBCD, F
    
    incf    yearsBCD, F
    movfw   yearsBCD
    andlw   0x0F
    sublw   0x0A
    btfss   STATUS,Z
    goto    intDone
    movfw   yearsBCD

    andlw   0x0F0
    addlw   0x010
    movwf   yearsBCD            ; This will work up to the last leap year record in the s/w
    
intDone        
    movfw   PCLATH_TEMP
    movwf   PCLATH
    swapf   STATUS_TEMP, W
    movwf   STATUS
    swapf   W_TEMP,F
    swapf   W_TEMP, W
    retfie
    


;*****************************************************************************        
;
;   Function :  getDaysInMonthBCD
;               returns the number of days in the month, in binary
;               No check done on leap year - this follows in another subroutine
;
;   Input:      monthsBCD 
;
;   Output:     W contains number of days, in binary
;
;*****************************************************************************        
getDaysInMonthBCD
    movfw   monthsBCD
    btfss   monthsBCD, 4    
    goto    gdim001
    addlw   0x0A
    
gdim001
    andlw   0x0F
    addwf   PCL, F
    retlw    0       ; Entry not used
    retlw   31      ; January    
    retlw   28      ; Feb
    retlw   31      ; March
    retlw   30      ; April      
    retlw   31      ; May        
    retlw   30      ; June       
    retlw   31      ; July       
    retlw   31      ; August     
    retlw   30      ; Sept       
    retlw   31      ; Oct        
    retlw   30      ; Nov        
    retlw   31      ; Dec        
    retlw    0       ; Entry not used
    retlw    0       ; Entry not used
gdim001_end
    retlw    0       ; Entry not used

    IF ( (gdim001 & 0x0FF) >= (gdim001_end & 0x0FF) )
        MESSG   "Table gdim001 overflow"
    ENDIF



;*****************************************************************************        
;
;   Function :  adjustForLeapYear
;               checks monthsBCD if the month is february and the year
;               is a leap year. This is valid until 2039
;
;   Input:      monthsBCD 0x01 .. 0x12
;
;   Output:     W contains adjusting factor - 0 or 1 ( for Feb + leapyear ) 
;
;*****************************************************************************        
adjustForLeapYear
    movfw   monthsBCD
    sublw   0x02                ; Is the current month Feb?
    btfss   STATUS, Z           ; Yes, so adjust
    retlw   0                   ; No, so no offset
    
    ; Convert year to binary in W
    movfw   yearsBCD
    andlw   0x0F
    movwf   intTmp
    movlw   0
    btfsc   yearsBCD, 4
    addlw   D'10'
    btfsc   yearsBCD, 5
    addlw   D'20'
    
    addwf   intTmp, F
    movfw   intTmp
    sublw   D'39'
    btfss   STATUS, C  ;39 - year. c clear on overflow
    clrf    intTmp
    movfw   intTmp
     
afly001
    addwf   PCL, F
    retlw    1   ; 2000 is a leap year      
    retlw    0       
    retlw    0       
    retlw    0       
    retlw    1       
    retlw    0       
    retlw    0       
    retlw    0       
    retlw    1       
    retlw    0       
    retlw    0       
    retlw    0       
    retlw    1       
    retlw    0       
    retlw    0       
    retlw    0       
    retlw    1       
    retlw    0       
    retlw    0       
    retlw    0       
    retlw    1       
    retlw    0       
    retlw    0       
    retlw    0       
    retlw    1       
    retlw    0       
    retlw    0       
    retlw    0       
    retlw    1       
    retlw    0       
    retlw    0       
    retlw    0       
    retlw    1       
    retlw    0       
    retlw    0       
    retlw    0       
    retlw    1       
    retlw    0       
    retlw    0       
afly001_end    
    retlw    0       

    IF ( (afly001 & 0x0FF) >= (afly001_end & 0x0FF) )
        MESSG   "Table afly001_end overflow"
    ENDIF



;*****************************************************************************        
;
;   Function :  initHardware
;               reconfigures the peripherals of the chip.
;               This ensure that, if ESD or bugs have caused the configuration
;               change, it is restored. This is called every hour.
;
;   Input:      None
;
;   Output:     None
;
;*****************************************************************************        
initHardware
    movlw   B'00111110'         ; Start the 32KHz osc
    movwf   T1CON

    clrf    TMR1L
    movlw   0x010
    movwf   TMR1H
    bsf     STATUS, RP0    
    bsf     PIE1, TMR1IE
    bcf     STATUS, RP0    
    bsf     T1CON, TMR1ON

    call    eepHWInit
    call    pcHWInit            ; Setup the RS232 port
    call    dspHWInit
    call    kbdInit             ; configure the pic to detect keypresses
    return
    


;*****************************************************************************        
;
;   Function :  Main
;               Main application loop
;
;   Input:      None.
;
;   Output:     N/A
;
;*****************************************************************************        
Main
    clrf    STATUS                ; Select bank 0
    clrf    INTCON                ; No interrupts
    clrf    PCLATH                ; Code is in first bank

    ; Clear all bank0 ram
    movlw   0x20
    movwf   FSR
clrMem
    clrf    INDF
    incf    FSR, F
    btfss   FSR, 7              ; Have we reached 0x80?
    goto    clrMem              ; - No, so continue with clear
 
    call    initHardware  

    call    dspPwrOn            ; supply power to LCD module
    
    call    uiWelcome          ; Show the welcome screen, for 2 seconds
    

    movlw   D'200'              ; Display welcome screen for 2s.
    call    uiWait10ms           ; This also gives 32KHz clock time to start
    
        
    bsf     INTCON, PEIE
    bsf     INTCON, GIE         ; Enable global interrupts

    ; Check the eeprom - initialise it if necessary
    call    eepCheck
    
power_down
    ; turn display off for a bit
    call    dspPwrOff
    
    call    kbdWaitNoKey
    
power_down2
    bcf     INTCON, RBIF
    bsf     INTCON, RBIE        ; Enable keyboard interrupts

    ; Ignore any interrupt events that occurred while we were using the device
    bcf     flags1, INT_KEYPRESS
    bcf     flags1, INT_RTC_TICK
    
    sleep
    nop

    bcf     INTCON, RBIE        ; I dont want keyboard interrupts now

    btfss   flags1, INT_RTC_TICK
    goto    testKeys
    
    bcf     flags1, INT_RTC_TICK
    
    ; Check to see if we have ticked over to a new hour
    ; If so, save it to eeprom
    bcf     INTCON, GIE         ; Disable global interrupts
    movfw   daysBCD
    movwf   daysBCDTmp
    movfw   monthsBCD
    movwf   monthsBCDTmp
    movfw   yearsBCD
    movwf   yearsBCDTmp
    movfw   hoursBCD
    movwf   hoursBCDTmp
    movfw   minutesBCD
    movwf   minutesBCDTmp
    bsf     INTCON, GIE         ; Enable global interrupts

    sublw   0x00
    btfss   STATUS, Z
    goto    testKeys
    
    ; Save rtc
    clrf    eepNum
    clrf    eepAddH
    movlw   0x04
    movwf   eepAddL
    movfw   daysBCDTmp
    movwf   eepData
    call    eepWrite
    movfw   monthsBCDTmp
    movwf   eepData
    call    eepWrite
    movfw   yearsBCDTmp
    movwf   eepData
    call    eepWrite
    movfw   hoursBCDTmp
    movwf   eepData
    call    eepWrite
    movfw   minutesBCDTmp
    movwf   eepData
    call    eepWrite

    ; check reminders
    
    movlw   high checkRemindersB1
    movwf   PCLATH
    call    checkRemindersB1

    ; reconfigure the hardware, to make sure 
    ; that ESD or bugs have not 'flipped bits'
    call    initHardware

testKeys
    btfss   flags1, INT_KEYPRESS
    goto    power_down2
    
    bcf     flags1, INT_KEYPRESS
    
    call    dspPwrOn
    
    ; First, display any active reminders
    
    movlw   0x01
    movwf   viewType
    movlw   high viewRemindersB1
    movwf   PCLATH
    call    viewRemindersB1
    

    movlw   MENU_FIRST_ENTRY
    movwf   menuItem
    
lk1
    call    uiMenu

wait_no_key
    call    kbdWaitNoKey

    clrf    kbdTimeout              ; This is an interrupt based timer
    
wait_key
    movlw   0x04
    subwf   kbdTimeout,W
    btfsc   STATUS, C               ; If C is set, it means kbdTimeout has exceed 4 ticks ( 1 minute )
    goto    power_down

    ; Wait for a key to be pressed 
    call    kbdReadRaw
    movfw   rawKey
    sublw   0x0f
    btfsc   STATUS, Z
    goto    wait_key

    call    kbdDebounce
    movfw   rawKey
    sublw   0x0f
    btfsc   STATUS, Z
    goto    wait_key
    
    ; OK, we have a key.
    ; Only navigation or the '#' key are accepted here
    
gotKey
    movfw   rawKey
    sublw   5
    btfsc   STATUS, Z
    goto    gotKey2          
     
    movfw   rawKey
    sublw   7
    btfsc   STATUS, Z
    goto    gotKey3
    
    movfw   rawKey
    sublw   0
    btfss   STATUS, Z
    goto    wait_no_key          ; Not a key I can accept, so continue
    
    goto    doMenuItem
    
gotKey3
    ; '2' Pressed - go up
    decf    menuItem, F 
    goto    lk1
  
gotKey2
    ; '8' Pressed - go up 
    incf    menuItem, F    
    goto    lk1

doMenuItem
    ; The '#' key was pressed, so action the MenuItem
    
    call    handleMenu
    
    goto    power_down




;*****************************************************************************        
;
;   Function :  handleMenu
;               Jumps to the function for a given menu number
;
;   Input:      None.
;
;   Output:     When the select routine returns, it will return to the function
;               that called handleMenu.
;
;*****************************************************************************        
handleMenu
    movlw   high hm001
    movwf   PCLATH
    movfw   menuItem
    andlw   0x0f

hm001
    addwf    PCL ,F
    goto    enterFuel
    goto    reportMonth
    goto    viewReminders
    goto    addReminder
    goto    DeleteReminder
    goto    setUnits
    goto    setClock
    goto    resetCounters
    goto    connectToPC
    goto    blankDisplay
    goto    Main
    goto    Main
    goto    Main
    goto    Main
    goto    Main
hm_end
    goto    Main

    IF ( (handleMenu & 0x0FF) >= (hm_end & 0x0FF) )
        MESSG   "Table handleMenu overflow"
    ENDIF

    ; Pull in all the other project Modules
    
    #include "menufunc.inc"
    #include "display.inc"
    #include "eeprom.inc"
    #include "pc.inc"
    #include "ui.inc"           ; ui.inc sets the code bank to 0x800 ...
    #include "calc.inc"         ; Therefore these funcs in bank1
    #include "menufb1.inc"


; The follow lines define the data to be programmed into the PIC's 
; internal eeprom. These are used to display the menu on the LCD
            
          ORG 0x2100 
Welcome 
    DE      "EPE Car computer"
Welcome2
    DE      "v1.0 5/10/03   "

Menu
    DE      "Enter Fuel      "   
    DE      "Report by Month "   
    DE      "View Reminders  "   
    DE      "Add Reminder    "   
    DE      "Delete Reminder "   
    DE      "Set Units       "   
    DE      "Set Clock       "   
    DE      "Erase memory    "   
    DE      "Connect to PC   "   
    DE      "Blank Display   "  
    
    END                ; End of program